/**
 * @fileOverview
 * @author <a href="mailto:simonsliu@tencent.com">simonsliu</a>
 * @date 12-2-15 - 下午4:39
 */
(function(window) {
	var emptyArray = Array.prototype;
	/**
	 * @name Skin.app.TabBar
	 * @class
	 */
	Cube.define('Skin.app.TabBar', {
		/**
		 * 类初始化完成
		 * @name Skin.app.TabBar#onInitialize
		 * @event
		 */
		/**
		 * 添加标签按钮点击
		 * @name Skin.app.TabBar#onAddTabButtonClick
		 * @event
		 */
		/**
		 * 添加标签按钮鼠标划过
		 * @name Skin.app.TabBar#onAddTabButtonMouseOver
		 * @event
		 */
		/**
		 * 标签上鼠标按下
		 * @name Skin.app.TabBar#onTabMouseDown
		 * @event
		 */
		/**
		 * 标签关闭按钮点击
		 * @name Skin.app.TabBar#onTabCloseButtonClick
		 * @event
		 */
		/**
		 * 删除标签前
		 * @name Skin.app.TabBar#onBeforeRemoveTab
		 * @event
		 */
		/**
		 * 删除标签
		 * @name Skin.app.TabBar#onRemoveTab
		 * @event
 		 */
		/**
		 * 鼠标滚轮滚动
		 * @name Skin.app.TabBar#onMouseWheel
		 * @event
		 */
		/**
		 * 标签栏鼠标双击
		 * @name Skin.app.TabBar#onDblClick
		 * @event
		 */
		/**
		 * 创建标签
		 * @name Skin.app.TabBar#onCreateTab
		 * @event
		 */
		config: {
			element: null,                  // 最外层元素
			maxTabWidth: 213,               // 标签最大宽度
			minShowCloseButtonWidth: 35,    // 最小显示关闭按钮尺寸
			minShowIconWidth: 42,           // 最小显示icon尺寸
			minShowTitleWidth: 62,          // 最小显示标题尺寸
			createAnimTime: 100,            // 新建标签动画播放时长
			removeAnimTime: 100,            // 移除标签动画播放时长
			relayoutAnimTime: 100,          // 调整标签宽度动画播放时长
			animEasing: 'easeInOutSine',           // 动画播放算法
			noAnimChange: 4,                // 取消动画的最小变化量
			enableAnimate: true,            // 启用动画
			onAddTabButtonClick: null,
			onAddTabButtonMouseOver: null,
			onBeforeRemoveTab: null,
			onCreateTab: null,
			onDblClick: null,
			onInitialize: null,
			onMouseWheel: null,
			onRemoveTab: null,
			onContextMenu: null,
			onTabCloseButtonClick: null,
			onTabCloseButtonMouseOver: null,
			onTabContextMenu: null,
			onTabDblClick: null,
			onTabDeactivate: null,
			onTabMouseDown: null,
			onTabMouseOver: null,
			onTabMouseUp: null
		},
		/**
		 * 构造函数
		 * @param {Cube.Element} config
		 */
		constructor: function(config) {
			this.$initConfig(Cube.extend(this.config, config || {}));

			var self = this;

			// 构造HTML结构
			this.addTabButtonContainer = Cube('<div>')
				.addClass('add-tab-button-container')
				.appendTo(this.element);

			this.addTabButton = Cube('<button>')
				.addClass('add-tab-button')
				.appendTo(this.addTabButtonContainer);

			// 初始化数据
			this.oldTabWidth = this.tabWidth = this.maxTabWidth;
			this.maxTotalWidth = this.getMaxTotalWidth();
			// 绑定事件
			this.addTabButton
				.bind('click.tabBar', function() {
					if (self.onAddTabButtonClick && Cube.isFunction(self.onAddTabButtonClick)) {
						self.onAddTabButtonClick.call(self);
					}
				})
				.bind('mouseover.tabBar', function() {
					if (self.onAddTabButtonMouseOver && Cube.isFunction(self.onAddTabButtonMouseOver)) {
						self.onAddTabButtonMouseOver.call(self);
					}
				});

			this.element
				.bind('mouseout.tabBar', function(e) {
					var toElement = e.toElement;
					if (toElement !== self.element[0] &&
						Cube(toElement).parents().indexOf(self.element[0]) < 0) {
						self.mouseover = false;
						self.updateWidth(self.getMaxTotalWidth());
						self.releaseAddButton();
						self.relayout(self);
					}
				})
				.bind('mouseover.tabBar', function(e) {
					self.mouseover = true;
				})
				.bind('mousewheel.tabBar', function(e) {
					if (self.onMouseWheel && Cube.isFunction(self.onMouseWheel)) {
						self.onMouseWheel.call(self, e);
					}
				})
				.bind('dblclick.tabBar', function(e) {
					if (self.onDblClick && Cube.isFunction(self.onDblClick)) {
						self.onDblClick.call(self, e);
					}
				})
				.bind('contextmenu.tabBar', function(e) {
					if (self.onContextMenu && Cube.isFunction(self.onContextMenu)) {
						self.onContextMenu.call(self, e);
					}
				});

			this.length = 0;
			this.initialize();
		},

		/**
		 * 控件初始化
		 */
		initialize: function() {
			if (this.onInitialize && Cube.isFunction(this.onInitialize)) {
				this.onInitialize.call(this);
			}
		},

		/**
		 * 创建新标签
		 * @param {PlainObject} data tab标签数据 {@link Tab#constructor}
		 */
		createTab: function(data) {
			var self = this, tab, $tab, index = data.index, i,
				targetWidth,
				maxTotalWidth = this.getMaxTotalWidth(),
				enableAnimate;

			delete data.index;
			// 没有传入index值时默认为最后一个
			if (index !== 0 && !index) {
				index = this.length;
			}

			tab = new Tab(data, this);
			$tab = $(tab.element[0]);

			// 插入Tab对象到TabBar中
			if (index === 0) {
				this.unshift(tab);
			} else if (index === this.length) {
				this.push(tab);
			} else {
				for (i = this.length; i > index; i--) {
					this[i] = this[i - 1];
				}
				this[i] = tab;
				this.length++;
			}

			this.updateWidth(maxTotalWidth);

			enableAnimate = this.enableAnimate && this.tabWidth > this.noAnimChange;

			targetWidth = this.tabWidth + ( index < this.restWidth ? 1 : 0 );

			tab.element
				.css({
					width: '0px'
				})
				.insertBefore(this.element[0].children[index]);

			// 有动画时，在开始前更新组件显示状态，防止本该隐藏的组件在动画播放完成前显示
			// 无动画时，只执行这一次更新组件显示状态。
			this.updateTabComponentDisplay(tab);

			if (this.tabWidth < this.maxTabWidth) {
				this.fixAddButton(maxTotalWidth);
			}

			if (enableAnimate) {
				if (tab.animating === true) {
					$tab.stop();
				} else {
					tab.animating = true;
				}
				$tab.animate({
					width: targetWidth + 'px'
				}, {
					duration: this.createAnimTime,
					easing: this.animEasing,
					complete: function() {
						self.updateTabComponentDisplay(tab);
						tab.animating = false;
					}
				});
			}

			this.relayout(self);

			if (data.isActive) {
				tab.activate();
			}

			if (this.onCreateTab && Cube.isFunction(this.onCreateTab)) {
				this.onCreateTab.call(this, tab);
			}

			return tab;
		},

		/**
		 * 移除标签
		 * @param {Tab} tab 标签对象
		 */
		removeTab: function(tab) {
			if (this.onBeforeRemoveTab && Cube.isFunction(this.onBeforeRemoveTab)) {
				if (this.onBeforeRemoveTab.call(this, tab) === false) {
					return;
				}
			}
			var self = this, i, len,
				index = this.indexOf(tab), totalWidthBeforeRemove,
				$tab = $(tab.element[0]),
				maxTotalWidth = this.getMaxTotalWidth(),
				enableAnimate;

			// 用于删除最后一个标签且需要重新布局时计算标签目标宽度
			totalWidthBeforeRemove = this.tabWidth * this.length + this.restWidth;

			this.splice(index, 1);

			if (this.mouseover) {
				if (index === this.length) {
					this.updateWidth(totalWidthBeforeRemove);
				}
			} else {
				this.updateWidth(maxTotalWidth);
			}

			if (this.mouseover) {
				if (index !== this.length || this.tabWidth === this.maxTabWidth) {
					this.releaseAddButton();
				} else {
					this.fixAddButton(totalWidthBeforeRemove);
				}
			} else {
				if (this.tabWidth !== this.oldTabWidth && this.tabWidth === this.maxTabWidth){
					this.releaseAddButton();
				}
			}

			enableAnimate = this.enableAnimate && !this.addButtonFixed;

			if (enableAnimate) {
				tab.element
					.css({
						opacity: 0,
						pointerEvents: 'none'
					})
					.addClass('no-drag');
				$tab.animate({
					width: 1
				}, {
					duration: this.removeAnimTime,
					easing: this.animEasing,
					complete: function() {
						tab.destroy();
						delete tab;
					}
				});
			} else {
				tab.destroy();
				delete tab;
			}

			self.relayout(self);

			if (this.onRemoveTab && Cube.isFunction(this.onRemoveTab)) {
				this.onRemoveTab.call(this, tab);
			}
		},

		updateWidth: function(maxTotalWidth) {
			var self = this, tabWidth, restWidth,
				length = this.length;

			tabWidth = Math.floor(maxTotalWidth / length);

			if (tabWidth >= this.maxTabWidth) {
				tabWidth = this.maxTabWidth;
				restWidth = 0;
			} else {
				restWidth = maxTotalWidth - tabWidth * length;
			}

			this.oldRestWidth = this.restWidth || 0;
			this.oldTabWidth = this.tabWidth;
			this.tabWidth = tabWidth;
			this.restWidth = restWidth;
		},

		/**
		 * 重新布局
		 * @param {Tab[]} tabs 需要重新布局的标签对象
		 */
		relayout: function(tabs) {
			var self = this, tab, $tab,
				tabWidth = this.tabWidth,
				restWidth = this.restWidth,
				oldTabWidth = this.oldTabWidth,
				i, len = tabs.length,
				enableAnimate = this.enableAnimate && Math.abs(tabWidth - oldTabWidth) >= this.noAnimChange;

			if (enableAnimate) {
				for (i = 0; i < len; i++) {
					tab = tabs[i], $tab = $(tab.element[0]);
					if (tab.animating === true) {
						$tab.stop();
					} else {
						tab.animating = true;
					}
					$tab.animate({
						width: ( tabWidth + (this.indexOf(tab) < restWidth ? 1 : 0)) + 'px'
					}, {
						duration: this.relayoutAnimTime,
						easing: this.animEasing,
						complete: function(tab) {
							this.updateTabComponentDisplay(tab);
							tab.animating = false;
						}.bind(self, tab)
					});
				}
			} else {
				for (i = 0; i < len; i++) {
					tab = tabs[i]; $tab = $(tab.element[0]);
					if (tab.animating === true) {
						$tab.stop();
						tab.animating = false;
					}
					tab.element.width(tabWidth + (this.indexOf(tab) < restWidth ? 1 : 0));
					this.updateTabComponentDisplay(tab);
				}
			}
		},

		/**
		 * 获取标签总宽度
		 * @return {Number} 标签宽度
		 */
		getTotalWidth: function() {
			var totalWidth = 0, i, len;
			for (i = 0, len = this.length; i < len; i++) {
				totalWidth += this[i].element.outerWidth();
			}
			return totalWidth;
		},

		/**
		 * 重新排序
		 * @param {Tab} tab 将要排序的标签
		 * @param {Number} index 要排到的位置
		 */
		resort: function(tab, index) {
			var originalIndex = this.indexOf(tab), i;

			if (index < originalIndex) {
				for (i = originalIndex; i > index; i--) {
					this[i] = this[i-1];
				}
			} else {
				for (i = originalIndex; i < index; i++) {
					this[i] = this[i+1];
				}
			}
			this[i] = tab;
		},

		/**
		 * 根据数据值获取tab对象
		 * @param {String} prop 数据名称
		 * @param {*} value 数据值
		 * @return {Tab} tab对象
		 */
		getTabByData: function(prop, value) {
			var i, len, tabData;
			for (i = 0, len = this.length; i < len; i++) {
				tabData = this[ i ].data;
				if (tabData[ prop ] && tabData[ prop ] === value) {
					return this[ i ];
				}
			}
		},

		/**
		 * 固定新建标签按钮容器位置
		 */
		fixAddButton: function(maxTotalWidth) {
			if (this.addButtonFixed === true) {
				return;
			}
			var element = this.element,
				addTabButtonContainer = this.addTabButtonContainer,
				getComputedStyle = window.getComputedStyle,
				parseInt = window.parseInt;

			addTabButtonContainer.css({
				position: 'absolute',
				left: maxTotalWidth
					+ parseInt(getComputedStyle(element[0])['padding-left'].replace('px', ''), 10) || 0
					+ 'px'
			});
			this.addButtonFixed = true;
		},
		/**
		 * 释放新建标签按钮容器位置
		 */
		releaseAddButton: function() {
			if (this.addButtonFixed === false) {
				return;
			}
			this.addTabButtonContainer.css({
				position: 'relative',
				left: 'auto'
			});
			this.addButtonFixed = false;
		},

		/**
		 * 更新标签内组件的显示状态
		 * @param {Tab} tab 标签对象
		 */
		updateTabComponentDisplay: function(tab) {
			if (!tab) {
				return;
			}
			var minShowCloseButtonWidth = this.minShowCloseButtonWidth,
				minShowIconWidth = this.minShowIconWidth,
				minShowTitleWidth = this.minShowTitleWidth,
				tabWidth = this.tabWidth; // 不考虑余数以使所有标签的显示状态统一

			if (tabWidth <= minShowCloseButtonWidth) {
				tab.closeButton.shouldHidden = true;
				if (tab.data.isActive && !tab.closeButton.neverShow) {
					tab.showComponent('closeButton');
				} else {
					tab.hideComponent('closeButton');
				}
			} else {
				tab.closeButton.shouldHidden = false;
				if (!tab.closeButton.neverShow) {
					tab.showComponent('closeButton');
				}
			}

			if (tabWidth <= minShowIconWidth) {
				tab.hideComponent('iconCont');
				tab.iconCont.shouldHidden = true;
			} else {
				tab.showComponent('iconCont');
				tab.iconCont.shouldHidden = false;
			}

			if (tabWidth <= minShowTitleWidth) {
				tab.hideComponent('titleCont');
			} else {
				tab.showComponent('titleCont');
			}
		},
		/**
		 * 获取最大允许总宽度
		 * @return {Number} 当前允许的最大总宽度
		 */
		getMaxTotalWidth: function() {
			return this.element.width() - this.addTabButtonContainer.outerWidth();
		},
		indexOf: emptyArray.indexOf,
		splice: emptyArray.splice,
		slice: emptyArray.slice,
		unshift: emptyArray.unshift,
		push: emptyArray.push,
		shift: emptyArray.shift,
		pop: emptyArray.shift
	});


	/**
	 * Tab工厂
	 * @private
	 * @param {PlainObject} data 标签数据
	 * @param {String} data.title 页面标题
	 * @param {String} data.url 页面地址
	 * @param {String} data.icon 页面icon地址
	 * @param {String} data.iconCache 页面icon地址缓存，用于icon已经就绪但仍不能显示的临时存储
	 * @param {Boolean} data.isBookmarked 是否已收藏的标识
	 * @param {Number} data.tabState 标签栏状态代号
	 * @param {Number} data.isFresh
	 * @param {PlainObject} data.loadStatus 加载状态对象
	 * @param {Number} data.id 页面ID
	 * @param {Number} data.parentId 父页面ID
	 * @param {PlainObject} data.coreTypeNotify 内核相关信息
	 * @param {Boolean} data.isActive 标签是否激活
	 * @param {Boolean} data.urlIsInner 是否是内部页面
	 * @param {PlainObject} data.addressInfo 页面地址信息
	 * @param {Boolean} data.back 是否可后退
	 * @param {Boolean} data.forward 是否可前进
	 * @param {Boolean} data.urlSpoof url漏洞标识
	 * @param {Number} data.engine 引擎代号 0 -> IE内核，1 -> Webkit内核，2 -> 内部页
	 * @param {Boolean} data.error 是否有错误
	 * @param {Boolean} data.scripterror 是否有脚本错误
	 * @param {String} data.status 页面加载状态 loadend -> 加载完成
	 * @param {Number} data.mode 切换Tab的模式，0 -> 点击Tab切换；1 -> Ctrl+Tab切换
	 * @param {Boolean} data.accType 新建Tab的方式，true -> Ctrl+T新建；false -> 其它方式(点击新建按钮、打开新链接等)
	 * @param {Cube.app.tabbar.TabBar} tabBar 所属tabBar
	 */
	function Tab(data, tabBar) {
		var self = this,
			tabBar = this.tabBar = tabBar,
			name, element, prop;

		element = this.element = Cube('<div>')
			.addClass('tab');

		this.content = Cube('<div>')
			.addClass('content')
			.appendTo(this.element);

		this.iconCont = Cube('<div>')
			.addClass('icon')
			.appendTo(this.content);
		this.titleCont = Cube('<div>')
			.addClass('title')
			.appendTo(this.content);
		this.closeButton = Cube('<button>')
			.addClass('close-button')
			.appendTo(this.content);

		this.data = {};

		// 获取tab相关事件回调
		for (name in tabBar) {
			if (tabBar.hasOwnProperty(name) && /^onTab\w+$/.test(name)) {
				this[ name.replace('onTab', 'on') ] = tabBar[ name ];
			}
		}

		if (data['title']) {
			data['title'] = Cube.String.htmlEncode(data['title']);
		}

		this.setData(data);

		this.closeButton
			.bind('click.tab', function(event) {
				event.stopPropagation();
				if (self.onCloseButtonClick && Cube.isFunction(self.onCloseButtonClick)) {

					self.onCloseButtonClick.call(tabBar, event, self);
				}
			})
			.bind('mouseover.tab', function(event) {
				event.stopPropagation();
				if (self.onCloseButtonMouseOver && Cube.isFunction(self.onCloseButtonMouseOver)) {
					self.onCloseButtonMouseOver.call(tabBar, event, self);
				}
			});

		element
			.bind('mouseover.tab', function(event) {
				if (event.target !== self.closeButton[0]) {
					if (!self.closeButton.neverShow) {
						self.showComponent('closeButton');
					}
					if (self.onMouseOver && Cube.isFunction(self.onMouseOver)) {
						self.onMouseOver.call(tabBar, event, self);
					}
				}
			})
			.bind('mouseout.tab', function(event) {
				var toElement = event.toElement;
				if (toElement !== self.element[0] &&
					!Cube(toElement).parents().indexOf(self.element[0])) {
					if (self.closeButton.shouldHidden === true && !self.data.isActive) {
						self.hideComponent('closeButton');
					}
				}
			})
			.bind('mousedown.tab', function(event) {
				// 防止与点击关闭按钮及右键菜单冲突
				if (event.target !== self.closeButton[0] && event.which !== 3) {
					if (self.onMouseDown && Cube.isFunction(self.onMouseDown)) {
						self.onMouseDown.call(tabBar, event, self);
					}
				}
			})
			.bind('dblclick.tab', function(event) {
				if (self.onDblClick && Cube.isFunction(self.onDblClick)) {
					self.onDblClick.call(tabBar, event, self);
				}
			})
			.bind('contextmenu.tab', function(event) {
				if (self.onContextMenu && Cube.isFunction(self.onContextMenu)) {
					self.onContextMenu.call(tabBar, event, self);
				}
			})
			.bind('mouseup.tab', function(event) {
				if (self.onMouseUp && Cube.isFunction(self.onMouseUp)) {
					self.onMouseUp.call(tabBar, event, self);
				}
			});

		return this;
	}

	Cube.extend(Tab, {
		// 需要显示的数据，用于判断数据更新后是否需要刷新显示
		// 键为数据名称，值为该数据是否有更新的标识
		displayData: {
			'icon': false,
			'title': false
		},
		/**
		 * @private
		 * @member Tab
		 * 加载状态集
		 */
		loadingStages: ['before-loading', 'loading']
	});

	Tab.prototype = {
		destroy: function() {
			this.closeButton.unbind('.tab');
			this.element
				.unbind('.tab')
				.remove();
		},
		/**
		 * 设置标签数据
		 * @param {PlainObject} data 标签数据键值对象或键值
		 */
		setData: function(data) {
			console.log(data)
			var displayDataChange = false,
				displayData = Tab.displayData;
			for (var dataName in displayData) {
				if (displayData.hasOwnProperty(dataName) && data[dataName] && data[dataName] !== this.data[dataName]) {
					displayDataChange = true;
					displayData[ dataName ] = true;
				}
			}
			if (Cube.isPlainObject(data)) {
				Cube.extend(true, this.data, data);
			}

			return displayDataChange ? this.refresh() : this;
		},
		/**
		 * 刷新界面显示内容
		 */
		refresh: function() {
			var data = this.data, iconContElem = this.iconCont[0],
				displayData = Tab.displayData, img;

			if (data.icon && displayData.icon === true) {
				this.stopLoading();
				if (iconContElem.firstChild && iconContElem.firstChild.nodeName === 'IMG') {
					iconContElem.firstChild.src = data.icon;
					iconContElem.firstChild.style.display = '';
				} else {
					img = Cube('<img>')[0];
					img.src = data.icon;
					this.iconCont.append(img);
				}
				displayData.icon = false;
			}

			if (displayData.title === true) {
				this.titleCont[0].innerText = Cube.String.htmlDecode(data.title || '');
				displayData.title = false;
			}

			return this;
		},
		/**
		 * 显示组件
		 * @param {String} name 组件名称
		 */
		showComponent: function(name) {
			if (this[ name ].hidden === false) {
				return;
			}
			if (Cube.isString(name) && this[ name ] instanceof Cube.Element) {
				this[ name ][0].style.removeProperty('display');
				this[ name ].hidden = false;
			}
			return this;
		},
		/**
		 * 隐藏组件
		 * @param {String} name 组件名称
		 */
		hideComponent: function(name) {
			if (this[ name ].hidden === true) {
				return;
			}
			if (Cube.isString(name) && this[ name ] instanceof Cube.Element) {
				this[ name ].css({
					display: 'none'
				}).hidden = true;
			}
			return this;
		},
		/**
		 * 设置loading状态
		 * @param {String} stage loading阶段
		 */
		setLoading: function(stage) {
			if (stage && this[stage]) {
				return;
			}
			if (this.iconCont[0].firstChild) {
				this.iconCont[0].firstChild.style.display = 'none';
			}
			this.data.icon = '';
			this.iconCont.removeClass(Tab.loadingStages.join(' '));
			this.iconCont.addClass(stage);
			this[stage] = true;
		},
		/**
		 * 销毁loading
		 * @param {String} stage loading阶段
		 */
		stopLoading: function(stage) {
			if (stage && !this[stage]) {
				return;
			}
			this.iconCont.removeClass(stage || Tab.loadingStages.join(' '));
			stage ? (this[stage] = false)
				: Tab.loadingStages.forEach(function(s) {
					this[s] = false;
				}.bind(this));
		},
		/**
		 * 激活标签
		 */
		activate: function() {
			var tabBar = this.tabBar;
			// 放置重复激活
			if (tabBar.activeTab === this) {
				return;
			}
			if (this.onBeforeActivate && Cube.isFunction(this.onBeforeActivate)) {
				this.onBeforeActivate.call(this.tabBar, this);
			}
			if (tabBar.activeTab) {
				tabBar.activeTab.deactivate();
			}
			this.element.addClass('active');
			tabBar.activeTab = this;
			this.data.isActive = true;
			if (this.onActivate && Cube.isFunction(this.onActivate)) {
				this.onActivate.call(this.tabBar, this);
			}
		},
		deactivate: function() {
			this.element.removeClass('active');
			this.data.isActive = false;
			this.tabBar.activeTab = null;
			if (this.onDeactivate && Cube.isFunction(this.onDeactivate)) {
				this.onDeactivate.call(this.tabBar, this);
			}
		}
	}
})(window);
